<?php
/**
 * SCREETS © 2022
 *
 * SCREETS, d.o.o. Sarajevo. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and  accept  to the terms of the  License Agreement can install
 * and use this program.
 *
 * @package scxi
 * @author screets
 * @link https://screets.io
 */

/**
 * Hook the WordPress plugin into the appropriate WordPress actions and filters.
 *
 * @since 1.0.0
 */
class ScreetsChatAdmin {

  private $settings;

  /**
   * Constructor.
   *
   * @access public
   * @return void
   */
  public function __construct() {
    $this->settings = get_option('scxi_settings');

    add_action('admin_menu', array($this, 'adminMenu'));
    add_action('admin_init', array($this, 'settingsInit'));
    add_action('current_screen', array($this, 'includes'));
  }

  /**
   * Include admin files (conditionally).
   *
   * @since 3.0.0
   * @return void
   */
  public function includes () {
    if (!$screen = get_current_screen()) { 
      return; 
    }

    switch ($screen->id) {
      // check new updates
      case 'update':
      case 'update-core':
      case 'plugins':
      case 'plugin-install':
      case 'admin-ajax':
        require_once dirname( __FILE__ ) . '/lib/update-checker.php';
        new ScreetsChatUpdateChecker(dirname(plugin_basename(__DIR__)));
        break;
    }

    // console files
    if ($screen->id === 'toplevel_page_scxi_console') {
      // update user private token
      if (!empty($_GET['_scxireset'])) {
        delete_user_meta(get_current_user_id(), '_scxi_pkey');

        // redirect to remove token from URL to prevent refreshes
        wp_safe_redirect(admin_url('admin.php?page=scxi_console&_scxierr=unauth'));
        exit;
      }

       // update user private token
      if (!empty($_GET['_scxitoken'])) {
        $userid = get_current_user_id();

        if (metadata_exists('user', $userid, '_scxi_pkey')) {
          update_user_meta($userid, '_scxi_pkey', $this->_sanitizeUrlParam($_GET['_scxitoken']));
        } else {
          add_user_meta($userid, '_scxi_pkey', $this->_sanitizeUrlParam($_GET['_scxitoken']), true);
        }

        // redirect to remove token from URL to prevent refreshes
        wp_safe_redirect(admin_url('admin.php?page=scxi_console'));
        exit;
      }

      // register console shim file
      wp_register_script(
        'scxi-console',
        SCXI_FOLDER . '/static/js/console.shim.js',
        null,
        SCXI_VERSION
      );
      wp_enqueue_script('scxi-console');


    // chat settings files
    } elseif ($screen->id === 'settings_page_scxi_settings') {
      wp_register_style(
        'scxi-settings', 
        SCXI_FOLDER . '/static/css/settings.css', 
        null, 
        SCXI_VERSION
      );
      wp_enqueue_style('scxi-settings');

      wp_register_script(
        'scxi-settings',
        SCXI_FOLDER . '/static/js/settings.js',
        null,
        SCXI_VERSION
      );
      wp_enqueue_script('scxi-settings');
    }
  }

  /**
   * Admin menu.
   *
   * @since 3.0.0
   * @return void
   */
  public function adminMenu () {
    add_menu_page(
      SCXI_NAME, 
      SCXI_NAME, 
      'manage_options',
      'scxi_console', 
      array($this, 'renderConsole'),
      'data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHZpZXdCb3g9IjAgMCAyMjMuOTQ2IDIyMy45NDYiIGNsYXNzPSJfMnNFa0dyZXMgXzNGVDFpSjZpIF81aFhaczA5SCIgZGF0YS12LTJlMjZmMzllPSIiIGZpbGw9IndoaXRlIj48cGF0aCBkPSJNMjAwLjA2Miw1MC45NzNINjIuNzI5Yy0wLjkyNiwwLTEuODM5LDAtMi43MzgsMEg0LjQ5NGMtNC4wNjEsMC02LjEwNSw0Ljk2Ny0yLjk2NCw3LjU0MWwzNi40MTUsMjkuODg1djYwLjY1IGMwLDEzLjQzOSwxMS4zNDQsMjMuOTI0LDI0Ljc4MywyMy45MjRoMTM3LjMzM2MxMy40MzksMCwyMy44ODQtMTAuNDg1LDIzLjg4NC0yMy45MjRWNzQuODk3IEMyMjMuOTQ2LDYxLjQ1OCwyMTMuNTAxLDUwLjk3MywyMDAuMDYyLDUwLjk3M3ogTTE4MC4yNzksMTM1Ljk3M2gtOThjLTQuMTQzLDAtNy41LTMuMzU3LTcuNS03LjVzMy4zNTctNy41LDcuNS03LjVoOTggYzQuMTQzLDAsNy41LDMuMzU3LDcuNSw3LjVTMTg0LjQyMiwxMzUuOTczLDE4MC4yNzksMTM1Ljk3M3ogTTE4MC4yNzksMTAzLjk3M2gtNzVjLTQuMTQzLDAtNy41LTMuMzU3LTcuNS03LjVzMy4zNTctNy41LDcuNS03LjUgaDc1YzQuMTQzLDAsNy41LDMuMzU3LDcuNSw3LjVTMTg0LjQyMiwxMDMuOTczLDE4MC4yNzksMTAzLjk3M3oiIGRhdGEtdi0yZTI2ZjM5ZS1zPSIiPjwvcGF0aD48L3N2Zz4=',
      '26.2984'
    );

    // Settings
    add_options_page(SCXI_NAME, SCXI_NAME, 'manage_options', 'scxi_settings', array($this, 'renderSettings'));
  }

  /**
   * Render console page.
   *
   * @since 3.0.0
   * @return void
   */
  public function renderConsole () {
    $userid = get_current_user_id();

    echo '<script>
      window.screetsxci = { 
        appid: "'. esc_js(@$this->settings['appid']) .'", 
        uid: "' . $userid . '", 
        platform: "WP",
        privateKey: "' . esc_js(get_user_meta($userid, '_scxi_pkey', true)) . '",
        siteUrl: "' . $this->_getSiteDomain() . '",
        consoleUrl: "' . admin_url('admin.php?page=scxi_console') . '",
        settingsUrl: "' . admin_url('options-general.php?page=scxi_settings') . '",
        signinUrl: "' . SCXI_APP_URL . '/connectApp?platform=WP&method=signin&state=' . admin_url('admin.php?page=scxi_console') . '::' . wp_create_nonce('scxi-oauth') . '",
        bundleUrl: "' . esc_js(SCXI_FOLDER . '/core/bundle') . '",
        str: {
          setupError: "' . __('You should configure your real-time settings', 'scxi') . ' 👍",
          goSettings: "' . __('Go to chat settings', 'scxi') . '",
          invToken: "' . __('Invalid token. Please sign in again', 'scxi') . '",
          signin: "' . __('Sign in', 'scxi') . '",
        },
      };
    </script>';

    echo '<div id="scxi-console" class="scxi-console"></div>';
  }

  /**
   * Render settings page.
   *
   * @since 3.0.0
   * @return void
   */
  public function renderSettings () {
    $classname = get_called_class();
    $settingsUrl = admin_url('options-general.php?page=scxi_settings');

    ?>
    <form action='options.php' method='post' class="scxi-settings">
      <header class="scxi-header">
        <img src="<?php echo SCXI_FOLDER ?>/static/img/screets-logo-icon.png" alt="screets" class="scxi-logo">
        <div>
          <h1><?php echo SCXI_NAME; ?></h1>
          <h4>
            <strong><?php echo SCXI_BRAND_NAME . ' ' . SCXI_NAME; ?> </strong>
            <mark><?php echo SCXI_VERSION; ?></mark> &nbsp;-&nbsp;
            <a href="https://screets.com/chat/changelog" target="_blank"><?php _e('Changelog', 'scxi'); ?> &raquo;</a>
          </h4>
        </div>
      </header>

      <nav id="scxi-tabs" class="scxi-tabs nav-tab-wrapper">
        <a href="#general" class="scxi-tab-link nav-tab nav-tab-active" id="scxi-tab-link-general">
          <?php _e('General', 'scxi') ?>
        </a>
        <a href="#woocommerce" class="scxi-tab-link nav-tab" id="scxi-tab-link-woocommerce">
          <?php _e('WooCommerce', 'scxi') ?>
        </a>
        <a href="#msgs" class="scxi-tab-link nav-tab" id="scxi-tab-link-msgs">
          <?php _e('Messages', 'scxi') ?>
        </a>
      </nav>

      <div class="scxi-tab-container">
        <?php
        settings_fields('scxi-general');
        settings_fields('scxi-woocommerce');
        settings_fields('scxi-msgs');
        ?>

        <div id="scxi-tab-general" class="scxi-tab scxi-tab-active">
          <?php do_settings_sections('scxi-general') ?>
        </div>

        <div id="scxi-tab-woocommerce" class="scxi-tab">
          <?php do_settings_sections('scxi-woocommerce') ?>
        </div>

        <div id="scxi-tab-msgs" class="scxi-tab">
          <?php do_settings_sections('scxi-msgs') ?>
        </div>

        <?php submit_button() ?>
      </div>

    </form>
    <?php
  }

  /**
   * Initialize settings.
   *
   * @since 3.0.0
   * @return void
   */
  public function settingsInit () {
    register_setting('scxi-general', 'scxi_settings');
    register_setting('scxi-woocommerce', 'scxi_settings');
    register_setting('scxi-msgs', 'scxi_settings');

    // 
    // General tab
    // 
    add_settings_section(
      'scxi_section_db', 
      SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME, 
      array($this, 'sectionDesc'), 
      'scxi-general'
    );

    add_settings_field( 
      'scxi_appid', 
      __('Application ID', 'scxi'), 
      array($this, 'text'), 
      'scxi-general', 
      'scxi_section_db' ,
      array(
        'name' => 'appid',
        'placeholder' => __('Application ID', 'scxi'),
        'disabled' => true,
      )
    );

    add_settings_field( 
      'scxi_secret', 
      __('Identity verification secret', 'scxi'), 
      array($this, 'text'), 
      'scxi-general', 
      'scxi_section_db',
      array(
        'name' => 'secret',
        'placeholder' => __('Identity verification secret', 'scxi'),
        'desc' => sprintf(__('Find it under Settings > Security page on <a href="%s" target="_blank">chat console</a>.', 'scxi'), 'https://chat.screets.io'),
        'info' => __(' It helps to authorize your logged-in WordPress users securely.', 'scxi') . '<br>' . sprintf(__('<a href="%s" target="_blank">Learn more</a>', 'scxi'), 'https://docs.screets.io/chat/javascript-api/identity-verification.html')
      )
    );

    add_settings_field( 
      'scxi_token', 
      __('Access Token', 'scxi'), 
      array($this, 'text'), 
      'scxi-general', 
      'scxi_section_db',
      array(
        'name' => 'access_token',
        'placeholder' => __('Access token', 'scxi'),
        'desc' => sprintf(__('Find it under Settings > Security page on <a href="%s" target="_blank">chat console</a>.', 'scxi'), 'https://chat.screets.io'),
        'hint' => '<strong>(!) ' . __('Do NOT share your access token with anyone!', 'scxi') . '</strong>'
      )
    );

    // 
    // WooCommerce tab
    // 
    add_settings_section(
      'scxi_section_woocommerce', 
      'WooCommerce', 
      array($this, 'sectionDesc'), 
      'scxi-woocommerce'
    );

    add_settings_field( 
      'scxi_woocommerce_events', 
      __('Events', 'scxi'), 
      array($this, 'checkboxGroup'), 
      'scxi-woocommerce', 
      'scxi_section_woocommerce',
      array(
        'name' => 'woo_events',
        'desc' => '<strong>' . sprintf(__('In which events do you want data to be collected?', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME) . '</strong>',
        'values' => array(
          array(null, __('Orders', 'scxi')),
          array('woo_new_order', __('New order created', 'scxi'), array('disabled' => !class_exists('WooCommerce')) ),
          // array('woo_upd_order', __('Order updated', 'scxi'), array('disabled' => !class_exists('WooCommerce')) ),
        ),
      )
    );

    add_settings_field( 
      'scxi_woocommerce_fields', 
      __('Data Attributes', 'scxi'), 
      array($this, 'checkboxGroup'), 
      'scxi-woocommerce', 
      'scxi_section_woocommerce',
      array(
        'name' => 'woo_fields',
        'desc' => '<strong>' . sprintf(__('What data do you want to link to the chat?', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME) . '</strong>',
        'info' => sprintf(__('Choose the fields below you want to collect and bind to the related chat. Selected fields will be automatically created in %s after you save changes.', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME),
        'values' => array(
          array(null, __('Orders', 'scxi')),
          array('woo_new_order_id', __('Order ID', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_edit_link', __('Order Edit Link', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_items', __('Order Items', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_date', __('Order Date', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_total', __('Order Total (incl. taxes)', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_total_disc', __('Order Total Discount', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_billing', __('Order Billing', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_shipping', __('Order Shipping', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_billing_email', __('Customer Email', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_billing_phone', __('Customer Phone', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_payment_method', __('Payment Method', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_note', __('Customer Provided Note', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
          array('woo_new_order_coupon_codes', __('Order Coupon Codes', 'scxi'), array('disabled' => !class_exists('WooCommerce'))),
        ),
      )
    );

    

    // 
    // Messages tab
    // 
    $msgs = include(SCXI_PATH . '/core/i18n.php');

    add_settings_section(
      'scxi_section_msgs', 
      '', 
      array($this, 'sectionDesc'), 
      'scxi-msgs'
    );

    add_settings_field( 
      'scxi_use_ml', 
      '', 
      array($this, 'checkbox'), 
      'scxi-msgs', 
      'scxi_section_msgs' ,
      array(
        'name' => 'use_ml_plugin',
        'desc' => sprintf(__('Use multilingual plugin (i.e. %s)', 'scxi'), 'WPML, Polylang'),
        'info' => __('If enabled, "Strings translations" will be used instead of the fields below', 'scxi'),
      )
    );

    foreach ($msgs as $_item) {
      foreach ($_item as $key => $field) {
        add_settings_field( 
          'scxi_msg_' . $key, 
          $this->getSectionName($field[0]), 
          array($this, ($field[2] == false) ? 'text' : 'textarea'), 
          'scxi-msgs', 
          'scxi_section_msgs' ,
          array(
            'name' => '__' . $key,
            'placeholder' => $field[1],
            'hint' => $field[1],
          )
        );
      }
    }

    // 
    // Authorize the application
    // 
    if (isset($_GET['_scxistate']) && current_user_can('manage_options') && isset($_GET['a']) && isset($_GET['t'])) {
      // verify state
      if (wp_verify_nonce($_GET['_scxistate'], 'scxi-oauth')) {
        $userid = get_current_user_id();

        $settings = $this->settings;
        $settings['appid'] = $this->_sanitizeUrlParam($_GET['a']);
        $settings['secret'] = $this->_sanitizeUrlParam($_GET['secret']);

        // store access token
        if (metadata_exists('user', $userid, '_scxi_pkey')) {
          update_user_meta($userid, '_scxi_pkey', $this->_sanitizeUrlParam($_GET['t']));
        } else {
          add_user_meta($userid, '_scxi_pkey', $this->_sanitizeUrlParam($_GET['t']), true);
        }

        update_option('scxi_settings', $settings);

        // Refresh page
        wp_safe_redirect(admin_url('options-general.php?page=scxi_settings&authorized=1'));
      }
    }

    // 
    // Disconnect from screets Cloud
    // 
    if (isset($_GET['_scxidisconn'])) {
      $settings = $this->settings;
      $settings['appid'] = '';
      $settings['secret'] = '';
      
      update_option('scxi_settings', $settings);

      delete_user_meta(get_current_user_id(), '_scxi_pkey');

      // Refresh page
      wp_safe_redirect(admin_url('options-general.php?page=scxi_settings&disconnected=1'));
    }


    // 
    // Show notifications
    //
    if (isset($_GET['authorized'])) {
      add_action('admin_notices', array($this, '_noticeAuthorized'));
    }
    if (isset($_GET['disconnected'])) {
      add_action('admin_notices', array($this, '_noticeDisconn'));
    }
  }

  public function _noticeAuthorized () {
    $text = sprintf(__('Your WordPress successfully connected to the %s', 'scxi'), SCXI_CLOUD_NAME) . ' 🎉';
    $link ='<a href="' . admin_url('admin.php?page=scxi_console') . '" class="button button-primary">' . __('Go to chat console', 'scxi') . ' &raquo;</a>';

    echo "<div class=\"scxi-notice-authorized notice notice-success is-dismissible\" id=\"scxi-notice-authorized\">
      <h3>$text</h3>
      <p>$link</p>
    </div>";
  }

  public function _noticeDisconn () {
    $text = sprintf(__('Disconnected from %s successfully', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME);

    echo "<div class=\"scxi-notice-authorized notice notice-warning is-dismissible\" id=\"scxi-notice-authorized\">
      <p>$text</p>
    </div>";
  }

  /**
   * Get a section title by key name.
   *
   * @since 3.0.0
   * @return void
   */
  public function getSectionName ($key) {
    switch ($key) {
      case 'header': return __('Header', 'scxi');
      case 'topic': return __('Topics', 'scxi');
      case 'act': return __('Actions', 'scxi');
      case 'chat': return __('Chat', 'scxi');
      case 'prechat': return __('Pre-chat', 'scxi');
      case 'cnvs': return __('Conversations', 'scxi');
      case 'inbox': return __('Inbox', 'scxi');
      case 'ntf': return __('Notifications', 'scxi');
      case 'err': return __('Errors', 'scxi');
      case 'offline': return __('Offline form', 'scxi');
      case 'profile': return __('Profile fields', 'scxi');
      case 'privacy': return __('Privacy', 'scxi');
      case 'reply': return __('Composer', 'scxi');
      case 'general': return __('General', 'scxi');
      case 'rts': return __('Response time', 'scxi');
      case 'time': return __('Date', 'scxi');
      default:
        return $key;
    }
  }

  /**
   * Template: Render description of a section.
   *
   * @since 3.0.0
   * @return void
   */
  public function sectionDesc ($context) { ?>
    <div class="scxi-section-desc <?php echo str_replace('_', '-', $context['id']) ?>-desc">
      <?php 
        switch ($context['id']) {
          // Real-time database
          case 'scxi_section_db':
            echo '<p>'
               . sprintf(__('Connect your application with %s.', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_CLOUD_NAME)
               . '</p>';

            if (!empty($this->settings['appid'])) {
              echo '<p class="scxi-connected-box">'
                     . '<strong><span class="dashicons dashicons-saved"></span> '. __('Connected', 'scxi') .'</strong> &nbsp; ' 
                     . '<a id="scxi-disconnect-btn" href="' . admin_url('options-general.php?page=scxi_settings&_scxidisconn=1') . '" class="button button-small" data-confirm="' . __('Are you sure you want to disconnect? You can reconnect anytime.', 'scxi') . '">' . __('Disconnect', 'scxi') . '</a>'
                    . '</p>';

            } else {
              echo sprintf('<a href="%s" class="button">%s</a>', SCXI_APP_URL . '/connectApp?platform=WP&state=' . admin_url('admin.php?page=scxi_console') . '::' . wp_create_nonce('scxi-oauth') , __('Connect', 'scxi'));
            }
            break;

          // Data
          case 'scxi_section_woocommerce':
            echo '<p>' . sprintf(__('%s lets you create and track custom data attributes about your WooCommerce data, based on criteria that’s specific to your business.', 'scxi'), SCXI_BRAND_NAME . ' ' . SCXI_NAME) . '</p>';
            break;

          // Messages
          case 'scxi_section_msgs':
            echo '<p><em>' . __('Leave blank the fields you want to use the default message.', 'scxi') . '</em></p>';
            break;

        }
      ?>
    </div>
  <?php }

  /**
   * Template: Render HTML content.
   *
   * @since 3.0.0
   * @return void
   */
  public function html ($args) { ?>
    <?php echo $args['content'] ?>
  <?php }

  /**
   * Template: Render text field.
   *
   * @since 3.0.0
   * @return void
   */
  public function text ($args) { ?>
    <fieldset>
      <input 
        type="<?php echo (!isset($args['type'])) ? 'text' : $args['type'] ?>" 
        name="scxi_settings[<?php echo esc_attr($args['name']) ?>]" 
        id="scxi_settings_<?php echo esc_attr($args['name']) ?>"
        placeholder="<?php echo @esc_attr($args['placeholder']) ?>"
        value="<?php echo @esc_attr($this->settings[$args['name']]) ?>"
        class="regular-text <?php echo isset($args['disabled']) ? 'disabled' : '' ?>"
        <?php echo isset($args['disabled']) ? 'disabled="disabled"' : '' ?>
        >

      <?php if (isset($args['disabled'])): ?>
        <input type="hidden" name="scxi_settings[<?php echo esc_attr($args['name']) ?>]" value="<?php echo @esc_attr($this->settings[$args['name']]) ?>" />
      <?php endif; ?>

      <?php if (isset($args['desc'])): ?>
        <p class="description"><?php echo $args['desc'] ?></p>
      <?php endif ?>

      <?php if (isset($args['info'])): ?>
        <p class="description"><small class="scxi-info">
          <?php echo $args['info'] ?>
        </small></p>
      <?php endif ?>

      <?php if (isset($args['hint'])): ?>
        <p class="scxi-hint"><?php echo $args['hint'] ?></p>
      <?php endif ?>
    </fieldset>

  <?php }

  /**
   * Template: Render textarea field.
   *
   * @since 3.0.0
   * @return void
   */
  public function textarea ($args) { ?>
    <fieldset>
      <textarea 
        name="scxi_settings[<?php echo esc_attr($args['name']) ?>]" 
        id="scxi_settings_<?php echo esc_attr($args['name']) ?>"
        placeholder="<?php echo @esc_attr($args['placeholder']) ?>"
        class="large-text"
        rows="5"
        ><?php echo @esc_textarea($this->settings[$args['name']]); ?></textarea>

      <?php if (isset($args['desc'])): ?>
        <p class="description"><?php echo $args['desc'] ?></p>
      <?php endif ?>

      <?php if (isset($args['hint'])): ?>
        <p class="scxi-hint"><?php echo $args['hint'] ?></p>
      <?php endif ?>
    </fieldset>

  <?php }

  /**
   * Template: Render checkbox field.
   *
   * @since 3.0.0
   * @return void
   */
  public function checkbox ($args) { ?>
    <fieldset>
      <label>
        <input type="checkbox" name="scxi_settings[<?php echo esc_attr($args['name']) ?>]" <?php checked(@$this->settings[$args['name']], 1) ?> id="scxi_settings_<?php echo esc_attr($args['name']) ?>" value="1">

        <?php echo esc_html($args['desc']) ?>
      </label>

      <?php if (isset($args['info'])): ?>
        <p class="description"><small class="scxi-info">
          <?php echo $args['info'] ?>
        </small></p>
      <?php endif ?>

      <?php if (isset($args['hint'])): ?>
        <p class="scxi-hint"><?php echo $args['hint'] ?></p>
      <?php endif ?>
    </fieldset>

  <?php }

  /**
   * Template: Render a group of checkbox fields.
   *
   * @since 3.0.0
   * @return void
   */
  public function checkboxGroup ($args) { ?>
    <fieldset>
      <?php if (isset($args['desc'])): ?>
        <p class="description"><?php echo $args['desc'] ?></p>
      <?php endif ?>

      <?php if (isset($args['info'])): ?>
        <p class="description"><small class="scxi-info">
          <?php echo $args['info'] ?>
        </small></p>
      <?php endif ?>

      <?php if (isset($args['hint'])): ?>
        <p class="scxi-hint"><?php echo $args['hint'] ?></p>
      <?php endif ?>

      <?php
      $i = 0;
      while ($i < count($args['values'])): ?>
        <label class="<?php echo (isset($args['values'][$i][2]['disabled']) && $args['values'][$i][2]['disabled'] === true) ? 'scxi-field-disabled' : '' ?>">
          <?php if ($args['values'][$i][0] == null): ?>
            <div class="scxi-cb-group-title"><?php echo $args['values'][$i][1] ?></div>
          <?php else: ?>

            <input 
              type="checkbox" 
              name="scxi_settings[<?php echo esc_attr($args['name']) ?>][]"
              id="scxi_settings_<?php echo esc_attr($args['name']) ?>_<?php echo esc_attr($args['values'][$i][0]) ?>"
              value="<?php echo esc_attr($args['values'][$i][0]) ?>"
              <?php echo (isset($args['values'][$i]) && $this->settings && @$this->settings[$args['name']] !== null && in_array($args['values'][$i][0], $this->settings[$args['name']])) ? 'checked' : '' ?>
              <?php echo (isset($args['values'][$i][2]['disabled']) && $args['values'][$i][2]['disabled'] === true) ? 'disabled' : '' ?>
              >

            <?php echo $args['values'][$i][1] ?>
          <?php endif; ?>
        </label>
        
      <?php $i++; endwhile ?>
    </fieldset>

  <?php }

  /**
   * Get site domain (i.e. returns "screets.com").
   *
   * @since 3.0.0
   * @return string
   */
  private function _getSiteDomain () {
    $url = parse_url(get_site_url());

    if ($url == false && !isset($url['host'])) {
      return '';
    }

    return $url['host'];
  }

  /**
   * Sanitize URL parameter before storing into the database.
   *
   * @since 3.0.0
   * @return string
   */
  private function _sanitizeUrlParam ($val) {
    return wp_kses(trim($val), array());
  }
}