<?php
/*
  Plugin Name: screets Chat
  Plugin URI: https://screets.io
  Description: A beautiful chat widget for your website. Designed for <strong>sales</strong> & <strong>support</strong>.
  Version: 3.4.5
  Author: screets
  Author URI: https://screets.io
  Requires at least: 5.0
  Tested up to: 6.0.1
  Text Domain: scxi
  Domain Path: /languages
*/

/**
 * SCREETS © 2022
 *
 * SCREETS, d.o.o. Sarajevo. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and accept  to the terms of the  License Agreement can install
 * and use this program.
 *
 * @package scxi
 * @author screets
 * @link https://screets.io
 */

if ( ! defined( 'ABSPATH' ) ) exit;

const SCXI_VERSION = '3.4.5';
const SCXI_DEV = false;

/**
 * Hook the WordPress plugin into the appropriate WordPress actions and filters.
 *
 * @since 1.0.0
 */
final class ScreetsChat {
  /**
   * Product name.
   *
   * @since 3.0.0
   *
   * @type string
   */
  const NAME = 'screets Chat';

  /**
   * Uniquely identify plugin version.
   *
   * Bust caches based on this value
   *
   * @since 3.0.0
   *
   * @type string
   */
  const VERSION = SCXI_VERSION;

  /**
   * Unique domain of the plugin's translated text.
   *
   * @since 3.0.0
   *
   * @type string
   */
  const TEXT_DOMAIN = 'scxi';

  /**
   * Bind to hooks and filters
   *
   * @since 3.0.0
   * @return void
   */
  public static function init () {
    $classname = get_called_class();

    // load translated strings
    add_action('init', array($classname, 'loadTextDomain'));

    // make widgets available on front and back end
    add_action('widgets_init', array($classname, 'widgetsInit'));

    // register scripts
    add_action('wp_enqueue_scripts', array($classname, 'registerScripts'), 1, 0);

    // initialize for admin
    if (is_admin()) {
      add_action('init', array($classname, 'adminInit'));

    // render chat widget on front-end
    } else {
      add_action('init', array($classname, 'publicInit'));
      add_action('wp_footer', array($classname, 'render'), 200, 0);
    }

    // WooCommerce actions
    if (class_exists('WooCommerce')) {
      add_action('woocommerce_thankyou', array($classname, 'woo_newOrder'),  1, 1);
    }

    // shortcodes
    // static::registerShortcodeHandlers();
  }

  /**
   * Load translated strings for the current locale, if a translation exists.
   *
   * @since 3.0.0
   * @return void
   */
  public static function loadTextDomain () {
    load_plugin_textdomain(static::TEXT_DOMAIN, false, basename(dirname(__FILE__)) . '/languages/');

    // register strings for Polylang
    if (function_exists('pll_register_string')) {
      $msgs = include('core/i18n.php');

      foreach ($msgs as $_item) {
        foreach ($_item as $key => $field) {
          pll_register_string($key, $field[1], static::NAME);
        }
      }

    // register strings for WPML
    } elseif (has_action('wpml_register_single_string')) {
      $msgs = include('core/i18n.php');

      foreach ($msgs as $_item) {
        foreach ($_item as $key => $field) {
          do_action('wpml_register_single_string', static::NAME, $key, $field[1]);
        }
      }
    }
  }

  /**
   * Register widgets.
   *
   * @since 3.0.0
   * @return void
   */
  public static function widgetsInit () {}

  /**
   * Register common JavaScript files.
   *
   * @since 3.0.0
   * @return void
   */
  public static function registerScripts () {}

  /**
   * Hook into actions and filters specific to a WordPress administrative view.
   *
   * @since 3.0.0
   * @return void
   */
  public static function adminInit () {
    define('SCXI_NAME', 'Chat');
    define('SCXI_BRAND_NAME', 'screets');
    define('SCXI_CLOUD_NAME', 'Cloud');
    define('SCXI_PATH', untrailingslashit(plugin_dir_path(__FILE__)));
    define('SCXI_PLUGIN_FILE', plugin_basename( __FILE__ ));
    define('SCXI_EDITION', '');
    define('SCXI_FOLDER', plugin_dir_url(dirname( __FILE__ )) . 'screets-chat');
    define('SCXI_APP_URL', SCXI_DEV ? 'http://localhost:8080' : 'https://chat.screets.io');

    require_once(dirname( __FILE__ ) . '/core/admin.php');
    
    // Initiate admin
    new ScreetsChatAdmin();
  }

  /**
   * Register actions and filters shown in a non-admin context..
   *
   * @since 3.0.0
   * @return void
   */
  public static function publicInit () {
    // enhance web browser view only
    if (is_feed()) {
      return;
    }
  }

  /**
   * Get custom translations from options.
   */
  public static function getTranslations ($options) {
    $str = '';

    foreach ($options as $key => $value) {
      if (!empty($value) && substr($key, 0, 2) == '__') {
        $str .= '"' . substr($key, 2, strlen($key) - 2) . '": "' . esc_js($value) . "\",\n";
      }
    }

    return $str;
  }

  /**
   * Render widget.
   */
  public static function render () {
    $scxiOpts = '';
    $options = get_option('scxi_settings');

    if (is_null($options) || empty($options['appid'])) {
      return;
    }

    $str = '';

    // get translated strings from 3rd party plugin (i.e. WPML or Polylang)
    if (isset($options['use_ml_plugin']) && (function_exists('pll__') || has_filter('wpml_translate_single_string'))) {
      $msgs = include('core/i18n.php');

      // get translated string from Polylang
      if (function_exists('pll__')) {
        foreach ($msgs as $_item) {
          foreach ($_item as $key => $field) {
            $str .= '"' . $key . '": "' . pll__($field[1]) . "\",\n";
          }
        }
      
      // get translated string from WPML
      } elseif (has_filter('wpml_translate_single_string')) {

        // translate only if default language isn't the same with current language
        if (ICL_LANGUAGE_CODE !== apply_filters('wpml_default_language', null)) {
          foreach ($msgs as $_item) {
            foreach ($_item as $key => $field) {
              $str .= '"' . $key . '": "' . apply_filters('wpml_translate_single_string', $field[1], static::NAME, $key) . "\",\n";
            }
          }
        } else {
          $str = static::getTranslations($options);
        }
      }

    // get custom messages from options
    } else {
      $str = static::getTranslations($options);
    }

    if (is_user_logged_in() && !empty($options['secret'])) {
      $user = wp_get_current_user();

      $scxiOpts = 'userHash: "' . hash_hmac('sha256', $user->ID, $options['secret']) . '",
        uid: "' . $user->ID . '",
        email: "' . $user->user_email . '",
        name: "' . esc_js($user->user_nicename) . '",';
    }


    echo '<!-- screets Chat -->
    <script>
      window.screetsxi = {
        appid: "'. $options['appid'] .'",
        str: {' . $str . '},
        ' . $scxiOpts . '
      };
      (function(){var w = window;var i = function() { i.c( arguments ); };i.q = []; i.c = function( args ) { i.q.push(args); };w.scxi = i;w.addEventListener( "load", function() {var b = document.createElement("script");b.src = \'https://widget.screets.io/' . (SCXI_DEV ? 'dev' : '_') . '/\'+w.screetsxi.appid;b.async = !0;document.head.appendChild(b);}, !1 );})();
    </script>';
  }

  /**
   * WooCommerce actions.
   * 
   * @link https://stackoverflow.com/a/42533543/272478
   */
  public static function woo_newOrder ($orderId) {
    if (!$orderId) {
      return;
    }

    // Checking if this has already been done avoiding reload
    if (get_post_meta($orderId, 'scxi_woo_order_sent', true)) {
        return; // Exit if already processed
    }

    if (get_post_meta($orderId, 'scxi_sent', true)) {
      return; // exit if already sent to the chat server
    }

    $options = get_option('scxi_settings');

    if (!$options || !isset($options['woo_events']) || !in_array('woo_new_order', $options['woo_events']) || !isset($options['woo_fields']) || empty($options['appid']) || empty($options['access_token'])) {
      return;
    }

    $appid = explode('.', $options['appid']);

    if (count($appid) !== 2) {
      return; // not valid app id
    }
    
    $order = wc_get_order($orderId);
    $order_data = $order->get_data();

    $data = array('order.id' => $orderId);
    
    foreach ($options['woo_fields']  as $key => $value) {
      switch ($value) {
        case 'woo_new_order_id':
          $data['order.id'] = $orderId;
          break;

          case 'woo_new_order_edit_link':
            $data['order.edit_link'] = $order->get_edit_order_url();
            break;

          case 'woo_new_order_items':
            $items = array();

            foreach ($order->get_items() as $item_id => $item) {
              array_push($items, $item['name'] . ' (' . $item['quantity'] . ') - ' . wc_price($item->get_total()));
            }
            $data['order.items'] = $items;
            break;

          case 'woo_new_order_date': // timestamp
            $data['order.created_at'] = strtotime($order->get_date_created()->date('Y-m-d H:i:s'));
            break;

          case 'woo_new_order_total':
            $data['order.total'] = array(
              'curr' => $order->get_currency(),
              'v' => $order->get_total(),
            );
            break;

          case 'woo_new_order_total_disc':
            $data['order.total_discount'] = array(
              'curr' => $order->get_currency(),
              'v' => $order->get_total_discount(),
            );

            break;

          case 'woo_new_order_billing':
            $data['order.bill'] = $order->get_formatted_billing_address() . '<br>' . $order->get_billing_country();
            break;
            
            case 'woo_new_order_shipping':
              $data['order.shipping'] = $order->get_formatted_shipping_address() . '<br>' . $order->get_shipping_country();
            break;

          case 'woo_new_order_billing_email':
            $data['bill.email'] = $order->get_billing_email();
            break;
            
          case 'woo_new_order_billing_phone':
            $data['bill.tel'] = $order->get_billing_phone();
            break;
            
          case 'woo_new_order_payment_method':
            $data['bill.payment_method'] = $order->get_payment_method();
            break;

          case 'woo_new_order_note':
            $data['order.note'] = $order->get_customer_note();
            break;

          case 'woo_new_order_coupon_codes':
            $data['order.coupons'] = $order->get_coupon_codes();
            break;
      }
    }

    $data = array(
      'type' => 'chat',
      'group' => 'woo.order',
      'action' => 'startChat',
      'appid' => $appid[0],
      'chat_subj' => sprintf(__('WooCommerce Order #%s', 'scxi'), $orderId),
      'chat_msg' => (!empty($data['order.note']) ? $data['order.note'] : sprintf(__('New Order %s', 'scxi'), '#' . $orderId)),
      'user_email' => !is_user_logged_in() ? null : wp_get_current_user()->user_email,
      'data' => $data,
    );

    $url = SCXI_DEV ? 'http://localhost:9050/_/api' : 'https://scxi.' . $appid[1] . '.screets.io/_/api';

    // Get token
    $response = wp_remote_post($url . '/sendData', array(
      'method' => 'POST',
      'headers'     => array(
        'Authorization' => 'Bearer ' . $options['access_token'],
        'Content-Type'  => 'application/json',
    ),
      'body' => json_encode($data),
    ));
    if (!is_wp_error($response)) {
      $response = json_decode(wp_remote_retrieve_body($response), true);
  
      if (!empty($response['tk'])) {
        update_post_meta($orderId, 'scxi_woo_order_sent', esc_attr($orderId));
        
        // Send token to the chat server 
        // which will create a new chat for the user 
        // including the order
        echo '<script>window.addEventListener(\'load\', function () { scxi(\'_initChatByToken\', \'' . $response['tk'] . '\'); });</script>';
      }
    }
  }
}

// Run beautiful widget :)
add_action('plugins_loaded', array('ScreetsChat', 'init'), 0, 0);